import time

class BlackBody():
    def __init__(self, daemon, serial_number:str, threshold:float=1):
        self.daemon = daemon
        self.serial_number = serial_number
        res = self.daemon.send_command("open", [
            {"type": "string", "value": "BlackBody"},
            {"type": "dict", "value": {
                "serialNumber": serial_number,
        }}], 600)

        self.id = res["reference_id"]
        self.threshold = threshold

    def get_emiter_temperature(self):     
        res = self.daemon.send_command("execute", 
        [
            {"type": "string", "value": self.id},
            {"type": "string", "value": "get_emiter_temperature"},
            {"type": "list", "value": []}
        ])
        return res["temperature"]
    
    def get_background_temperature(self):     
        res = self.daemon.send_command("execute", 
        [
            {"type": "string", "value": self.id},
            {"type": "string", "value": "get_background_temperature"},
            {"type": "list", "value": []} 
        ])
        return res["temperature"]

    def set_abs_temperature(self, temperature:float, block=False):
        self.daemon.send_command("execute", 
        [
            {"type": "string", "value": self.id},
            {"type": "string", "value": "set_abs_temperature"},
            {"type": "list", "value": [
                {"type": "float", "value": temperature }
            ]} 
        ])

        if not block:
            return

        current = self.get_emiter_temperature()
        while not (temperature - self.threshold < current < temperature + self.threshold):
            print(f"BB waiting for ABS cmd to complete, target: {temperature}, current: {current}")
            time.sleep(1)
            current = self.get_emiter_temperature()
        print("BB target reached!.")

    def set_diff_temperature(self, temperature:float, block=False):
        self.daemon.send_command("execute", 
        [
            {"type": "string", "value": self.id},
            {"type": "string", "value": "set_diff_temperature"},
            {"type": "list", "value": [
                {"type": "float", "value": temperature }
            ]}
        ])

        if not block:
            return
            
        bg = self.get_background_temperature()
        emitter = self.get_emiter_temperature()
        diff = emitter - bg
        while not (temperature - self.threshold < diff < temperature + self.threshold):
            time.sleep(1)
            print(f"BB waiting for DIFF cmd to complete, diff-target: {temperature}, emitter: {emitter}, background: {bg}")
            bg = self.get_background_temperature()
            emitter = self.get_emiter_temperature()
            diff = emitter - bg
        print("BB target reached!.")
        
    def stop_emitter(self):
        self.daemon.send_command("execute", 
        [
            {"type": "string", "value": self.id},
            {"type": "string", "value": "stop_emitter"},
            {"type": "list", "value": [
            ]} 
        ])

    def get_info(self):
        return self.daemon.send_command("execute", 
        [
            {"type": "string", "value": self.id},
            {"type": "string", "value": "get_info"},
            {"type": "list", "value": [
            ]} 
        ])
    
class DummyBlackBody:
    def __init__(self, serial_number, threshold=1.0):
        self.serial = serial_number
        self.threshold = threshold
        self.mode = None
        self.abs_temp = 25.0
        self.diff_temp = 0.0
        self.bg_temp = 20.0

    def set_abs_temperature(self, temp, block=False):
        self.mode = "absolute"
        self.abs_temp = temp
        if block:
            self._simulate_delay()

    def set_diff_temperature(self, temp, block=False):
        self.mode = "differential"
        self.diff_temp = temp
        self.abs_temp = self.bg_temp + temp
        if block:
            self._simulate_delay()

    def stop_emitter(self):
        self.mode = "standby"
        self.abs_temp = self.bg_temp

    def get_emiter_temperature(self):
        return self.abs_temp

    def get_background_temperature(self):
        return self.bg_temp

    def get_info(self):
        return {
            "serial": self.serial,
            "model": "DummyBB-3000",
            "version": "1.0.0",
            "absolute-range": "-10 to 80",
            "differential-range": "-10 to 40"
        }

    def _simulate_delay(self):
        time.sleep(0.1)