import os, sys, threading, subprocess, shutil, calendar, binascii, time
from time import sleep
from ctypes import *
from datetime import datetime

from app.utility import *


class Bidentifier(object):

    def __init__(self, manager):
        self.m = manager
        self.last_cmd = time.monotonic()
        self.last_status = time.monotonic()
        self.base_obj = self.get_base_object()

        # Dislay
        self.last_display = time.monotonic()
        self.prev_clients = False

        # Status block
        self.block = False
        self.last_block = time.monotonic()

        threading.Thread(target=self.auto_10s, daemon=True).start()
        threading.Thread(target=self.auto, daemon=True).start()

        printout("Bidentifier: Initialized...")

    def auto_10s(self):
        while not self.m.exit:
            try:
                # Refresh GPS coordinates
                lat, longt = self.get_gps_position()
                if lat != 0.00 and longt != 0.00:
                    self.base_obj['GPS']['position'] = [lat, longt]
                else:
                    self.base_obj['GPS']['position'] = []

                if self.m.ws_handle:
                    printout("Bidentifier: Clients (Websocket) - {}".format(len(self.m.ws_handle.clients)))
                    printout("Bidentifier: Clients (Websocket): {}".format(self.m.ws_handle.has_clients))

            except Exception as e:
                printout(str(e))
                printout(traceback.format_exc())

            sleep(10)

    def auto(self):
        while not self.m.exit:
            try:

                # Release block status after timeout
                if self.block == True:
                    block_time = time.monotonic() - self.last_block
                    if block_time > 9:
                        self.block = False
                        self.last_block = time.monotonic()

                # Get all statuses
                if self.block == False:
                    status_time = time.monotonic() - self.last_status
                    if status_time > 0.8:
                        safety_time = time.monotonic() - self.last_cmd
                        if safety_time > 0.1:
                            self.get_status()

                            # Send to websocket clients
                            if self.m.ws_handle:
                                if self.m.ws_handle.has_clients:
                                    self.m.ws_handle.send_all({ "channel": "APP.UPDATE", "data": self.m.bidentifier.get_app_update() })

                        self.last_status = time.monotonic()

                # Read buffer
                res = self.send_device_cmd("\x67\x65\x74")
                self.receive_device_cmd(res)

                # Display Auto On (when all clients disconnect)
                off_display = time.monotonic() - self.last_display
                if off_display > 4:
                    if self.m.ws_handle:

                        # Display ON when clients 0
                        if self.m.ws_handle.has_clients == False:
                            if self.prev_clients != self.m.ws_handle.has_clients:
                                self.set_display(True)
                                self.prev_clients = self.m.ws_handle.has_clients

                        else:
                            self.prev_clients = self.m.ws_handle.has_clients

                    self.last_display = time.monotonic()

            except Exception as e:
                printout(str(e))
                printout(traceback.format_exc())

            sleep(0.1)

    def block_status(self):
        try:
            self.block = True
            self.last_block = time.monotonic()

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def send_device_cmd(self, cmd):
        try:
            self.last_cmd = time.monotonic()

            printout("BDR (Sent): {}".format(cmd))
            output = subprocess.run('echo -n -e "{}" | socat unix-connect:/var/run/webserver.socket STDIO'.format(cmd), shell=True, stdout=subprocess.PIPE)
            printout("BDR (Received): {}".format(binascii.hexlify(output.stdout).decode('latin1')))

            return output.stdout

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())
            return ""

    def receive_device_cmd(self, res):
        try:
            # Return if no packet
            if len(res) == 0: return

            if res[0] == 0xE6:
                pckt_len = res[1]
                if len(res) == pckt_len:
                    cmd = res[2]

                    # Get Brightness
                    # E6 07 56 [brt_val] [min] [max] ChS
                    if cmd == 0x56:
                        self.base_obj['BRT']['current'] = get_value_brt(res[3])

                    # Get Contrast
                    # E6 07 58 [ctrs_val] [min] [max] ChS
                    elif cmd == 0x58:
                        self.base_obj['CONT']['current'] = get_value_cnt(res[3])

                    # Get Weather
                    # E6 05 59 [weather_type] ChS
                    elif cmd == 0x59:
                        if res[3] == 0x01:
                            self.base_obj['WEATHER']['current'] = "hot"
                        elif res[3] == 0x02:
                            self.base_obj['WEATHER']['current'] = "normal"
                        elif res[3] == 0x03:
                            self.base_obj['WEATHER']['current'] = "wet"

                    # Get Filter
                    # E6 05 5A [filter_type] ChS
                    elif cmd == 0x5A:
                        if res[3] == 0x01:
                            self.base_obj['FILTER']['current'] = "on"
                        elif res[3] == 0x02:
                            self.base_obj['FILTER']['current'] = "off"

                    # Get Palette
                    # \e6\05\32\01\d0
                    elif cmd == 0x32:
                        if res[3] == 0x01:
                            self.base_obj['PALETTE']['current'] = "BH"
                        elif res[3] == 0x02:
                            self.base_obj['PALETTE']['current'] = "WH"
                        elif res[3] == 0x03:
                            self.base_obj['PALETTE']['current'] = "RH"
                        elif res[3] == 0x04:
                            self.base_obj['PALETTE']['current'] = "Rainbow"
                        elif res[3] == 0x05:
                            self.base_obj['PALETTE']['current'] = "Ironbow"
                        elif res[3] == 0x06:
                            self.base_obj['PALETTE']['current'] = "Isotherm"

                    # Get NUC Type
                    # \e6\05\31\03\d1
                    elif cmd == 0x31:
                        if res[3] == 0x02:
                            self.base_obj['NUC_TYPE']['current'] = "INT"
                        elif res[3] == 0x03:
                            self.base_obj['NUC_TYPE']['current'] = "EXT"

                    # Get Battery Level
                    # E6 05 5B [batt_lvl] ChS - Batt lvl. range[1 ~ 7]
                    elif cmd == 0x5B:
                        self.base_obj['BATT']['current'] = res[3]

                    # Get GPS Status
                    # \e6\06\61\01\01\81
                    elif cmd == 0x61:
                        if res[3] == 0x01:
                            self.base_obj['GPS']['current'] = "ON"
                        elif res[3] == 0x02:
                            self.base_obj['GPS']['current'] = "OFF"

                    # Get LRF / DMC / Target Lock
                    # \e6\07\69\01\01\01\89
                    elif cmd == 0x69:
                        lrf = res[3]
                        dmc = res[4]
                        trg = res[5]

                        if dmc == 0x00:
                            self.base_obj['DMC']['current'] = "OFF"
                        elif dmc == 0x01:
                            self.base_obj['DMC']['current'] = "ON"

                        if trg == 0x00:
                            self.base_obj['TARGET_LOCK']['current'] = "OFF"
                        elif trg == 0x01:
                            self.base_obj['TARGET_LOCK']['current'] = "ON"

                    # Get LRF data
                    # Response <hex> -
                    #  1|<E6>
                    #  2|<0C>
                    #  3|<5F>
                    #  4|<00>
                    #  5|<01>  //Unit - 1/Meter , 2/Yard - индикира избрания режим
                    #  6|<01>  \\__ Dist 1 0x016A   - 362m (винаги са в метри)
                    #  7|<6A>  //
                    #  8|<00>  \\__ Dist 2 0x007D   - 125m (винаги са в метри)
                    #  9|<7D>  //
                    # 10|<00>  \\__ Dist 3 0x00F0   - 240m (винаги са в метри)
                    # 11|<F0>  //
                    # 12|<52>  //ChSum
                    elif cmd == 0x5F:
                        unit = res[4]
                        if unit == 0x01:
                            self.base_obj['LRF']['UNT']['current'] = "METER"
                        elif unit == 0x02:
                            self.base_obj['LRF']['UNT']['current'] = "YARD"

                        distance1 = bytes_to_int([res[5], res[6]])
                        distance2 = bytes_to_int([res[7], res[8]])
                        distance3 = bytes_to_int([res[9], res[10]])
                        self.base_obj['LRF']['distances'] = [distance1, distance2, distance3]

                        self.m.ws_handle.send_all({ "channel": "LRF.DISTANCE.DONE", "data": { "distances": self.base_obj['LRF']['distances'] } })
                        self.block = False

                    # Get Auto NUC Interval
                    # E6 05 30 <minutes> 00
                    elif cmd == 0x30:
                        interval = res[3]

                    # Get Zoom
                    # e6 06 a1 00 <state> <zoom factor> <CHSUM>
                    elif cmd == 0xA1:
                        # State
                        if res[4] == 0x01:
                            self.base_obj['ZOOM']['current'] = "ZOOM_1"
                        elif res[4] == 0x02:
                            self.base_obj['ZOOM']['current'] = "ZOOM_2"
                        elif res[4] == 0x04:
                            self.base_obj['ZOOM']['current'] = "ZOOM_4"

                        # Zoom factor
                        # self.base_obj['ZOOM']['zoom_factor'] = round((res[5] * 0.025) + 1, 2)

                    # Pin Commands
                    elif cmd == 0xA0:

                        # Auth Pin
                        # <Е6> <LN> <A0> <CMD> <Status> <CHSUM>
                        if res[3] == 0x00:
                            state = "WRONG_PIN"
                            if res[4] == 0x01:
                                subprocess.getoutput('bdrctl --dlna start')
                                state = "OK"

                            elif res[4] == 0x02:
                                subprocess.getoutput('bdrctl --dlna stop')
                                state = "WRONG_PIN"

                            elif res[4] == 0x03:
                                subprocess.getoutput('bdrctl --dlna stop')
                                state = "TOO_MANY_FAILED_ATTEMPTS"

                            self.m.ws_handle.send_all({ "channel": "AUTH.CHECK.DONE", "data": { "STATE": state } })
                            self.block = False

                        # Change Pin
                        # e6 06 a0 04 01 44
                        elif res[3] == 0x04:
                            state = "FAILED"
                            if res[4] == 0x01:
                                state = "OK"

                            elif res[4] == 0x02:
                                state = "FAILED"

                            self.m.ws_handle.send_all({ "channel": "AUTH.CHANGE_PIN.DONE", "data": { "STATE": state } })
                            self.block = False

                    # Get Status
                    # Response <hex> - Get Status - доработени по тази схема.
                    # 1 |<E6>  // Start Byte (Тук ще друг - <Е6> е за PC)
                    # 2 |<14>  // Command Lenght
                    # 3 |<5C>  // Command code response
                    # 4 |<00>  // Sub Op Code
                    # 5 |<32>  // Brightness    (както се разбрахме не подавам min/max границите, но ако искате може да ги добавя)
                    # 6 |<4E>  // Contrast      ( ---//--- )
                    # 7 |<02>  // Filter - 1/On, 2/Off
                    # 8 |<02>  // Weather- 1/Hot  , 2/Normal, 3/Wet
                    # 9 |<02>  // Palette- 1/BH   , 2/WH    , 3/RedH, 4/Rainbow, 5/Ironbow, 6/(Isotherm - обвързан е със следващата команда)
                    # 10 |<91>  // Isotherm Level - [1] ... [241] value
                    # 11 |<06>  // PWRStatus - 1/Shutdown, 2/LowBatt, 3/25%, 4/50%, 5/75%, 6/100%, 7/External Power
                    # 12 |<02>  // NUC Type -  2/(Internal - обвързан е със следващата команда), 3/External
                    # 13 |<02>  // Internal -  1/Manual NUC, 2/(Auto NUC - обвързан е със следващата команда)
                    # 14 |<01>  // Auto NUC Interval [ minutes ]
                    # 15 |<01>  // DZoom <state>  - 1/ CONT_1,  2/ZOOM_X_2,  4/ZOOM_X_4    **
                    # 16 |<02>  // GPS Status  - 1/ON     , 2/OFF
                    # 17 |<02>  // DMC Status  - 1/ON     , 0/OFF
                    # 18 |<02>  // Target Lock - 1/ON     , 0/OFF
                    # 19 |<00>  // LRF Status  - 1/Engaged(Ready) , 0/Sleep
                    # 20 |<28>  // <zoom factor>  - Zoom factor: [0 : 40] / Continuous Zoom =  1 + <zoom factor> * 0.025  **
                    # 21 |<xx>  // ChSum
                    elif cmd == 0x5C:
                        # Brightness
                        self.base_obj['BRT']['current'] = get_value_brt(res[4])

                        # Contrast
                        self.base_obj['CONT']['current'] = get_value_cnt(res[5])

                        # Filter
                        if res[6] == 0x01:
                            self.base_obj['FILTER']['current'] = "on"
                        elif res[6] == 0x02:
                            self.base_obj['FILTER']['current'] = "off"

                        # Weather
                        if res[7] == 0x01:
                            self.base_obj['WEATHER']['current'] = "hot"
                        elif res[7] == 0x02:
                            self.base_obj['WEATHER']['current'] = "normal"
                        elif res[7] == 0x03:
                            self.base_obj['WEATHER']['current'] = "wet"

                        # Palette
                        if res[8] == 0x01:
                            self.base_obj['PALETTE']['current'] = "BH"
                        elif res[8] == 0x02:
                            self.base_obj['PALETTE']['current'] = "WH"
                        elif res[8] == 0x03:
                            self.base_obj['PALETTE']['current'] = "RH"
                        elif res[8] == 0x04:
                            self.base_obj['PALETTE']['current'] = "Rainbow"
                        elif res[8] == 0x05:
                            self.base_obj['PALETTE']['current'] = "Ironbow"
                        elif res[8] == 0x06:
                            self.base_obj['PALETTE']['current'] = "Isotherm"

                        # Isotherm Level
                        self.base_obj['PALETTE']['isotherm']['current'] = res[9]

                        # Battery
                        self.base_obj['BATT']['current'] = res[10]

                        # NUC Type
                        if res[11] == 0x02:
                            self.base_obj['NUC_TYPE']['current'] = "INT"
                        elif res[11] == 0x03:
                            self.base_obj['NUC_TYPE']['current'] = "EXT"

                        # Zoom
                        if res[14] == 0x01:
                            self.base_obj['ZOOM']['current'] = "ZOOM_1"
                        elif res[14] == 0x02:
                            self.base_obj['ZOOM']['current'] = "ZOOM_2"
                        elif res[14] == 0x04:
                            self.base_obj['ZOOM']['current'] = "ZOOM_4"

                        # GPS
                        if res[15] == 0x01:
                            self.base_obj['GPS']['current'] = "ON"
                        elif res[15] == 0x02:
                            self.base_obj['GPS']['current'] = "OFF"

                        # DMC
                        if res[16] == 0x01:
                            self.base_obj['DMC']['current'] = "ON"
                        elif res[16] == 0x00:
                            self.base_obj['DMC']['current'] = "OFF"

                        # TARGET LOCK
                        if res[17] == 0x01:
                            self.base_obj['TARGET_LOCK']['current'] = "ON"
                        elif res[17] == 0x00:
                            self.base_obj['TARGET_LOCK']['current'] = "OFF"

                        # LRF
                        if res[18] == 0x01:
                            self.base_obj['LRF']['current'] = "WAKE"
                        elif res[18] == 0x00:
                            self.base_obj['LRF']['current'] = "SLEEP"

                        # ZOOM FACTOR
                        # self.base_obj['ZOOM']['zoom_factor'] = round((res[19] * 0.025) + 1, 2)

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())
            return ""

    def get_app_update(self):
        try:
            return self.base_obj

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

            return {}

    def set_stream(self, enable):
        try:
            subprocess.getoutput("bdrvc_client http-stream {}".format(enable))

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def snapshot(self):
        try:
            self.send_device_cmd("e60663050100")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def recording(self, enable):
        try:
            if enable:
                self.send_device_cmd("e60663010100")

            else:
                self.send_device_cmd("e60663010200")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    # min = 0xCE (206), max = 0x32 (50)
    def set_brightness(self, brt):
        try:
            self.send_device_cmd("e60555{}00".format(format(brt & 0xFF, '02x')))
            self.base_obj['BRT']['current'] = brt

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_brightness(self):
        try:
            # E6 05 56 00 ChS
            self.send_device_cmd("e605560000")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    # min = 0x20 (32), max = 0x64 (100)
    def set_contrast(self, cont):
        try:
            self.send_device_cmd("e60557{}00".format(format(cont & 0xFF, '02x')))
            self.base_obj['CONT']['current'] = cont

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_contrast(self):
        try:
            # E6 05 58 00 ChS
            self.send_device_cmd("e605580000")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def set_weather(self, weather):
        try:
            if weather == "hot":
                self.send_device_cmd("e605590100")

            elif weather == "normal":
                self.send_device_cmd("e605590200")

            elif weather == "wet":
                self.send_device_cmd("e605590300")

            self.base_obj['WEATHER']['current'] = weather

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_weather(self):
        try:
            # E6 05 59 00 ChS
            self.send_device_cmd("e605590000")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def set_filter(self, filter):
        try:
            if filter == "on":
                self.send_device_cmd("e6055a0100")

            elif filter == "off":
                self.send_device_cmd("e6055a0200")

            self.base_obj['FILTER']['current'] = filter

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_filter(self):
        try:
            # E6 05 5A 00 ChS
            self.send_device_cmd("e6055a0000")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def set_palette(self, palette):
        try:
            if palette == "WH":
                self.send_device_cmd("e6050c0000")

            elif palette == "BH":
                self.send_device_cmd("e6050d0000")

            self.base_obj['PALETTE']['current'] = palette

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_palette(self):
        try:
            # E6 05 32 00 ChS
            self.send_device_cmd("e605320000")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def do_nuc(self):
        try:
            self.send_device_cmd("e6050e0000")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def set_nuc_type(self, nuc_type):
        try:
            if nuc_type == "INT":
                self.send_device_cmd("e605310200")

            elif nuc_type == "EXT":
                self.send_device_cmd("e605310300")

            self.base_obj['NUC_TYPE']['current'] = nuc_type

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_nuc_type(self):
        try:
            # E6 05 31 00 ChS
            self.send_device_cmd("e605310000")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def set_standby(self, standby):
        try:
            if standby:
                self.send_device_cmd("e605520200")
                self.base_obj['STANDBY']['current'] = "ON"

            else:
                self.send_device_cmd("e605520300")
                self.base_obj['STANDBY']['current'] = "OFF"

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    # {"mnt": 3, "year": 2024, "day": 13}
    def set_calendar(self, cal):
        try:
            now = datetime.now()
            hour = str(now.hour).zfill(2)
            minute = str(now.minute).zfill(2)

            subprocess.getoutput('date --set "{}-{}-{} {}:{}:00"'.format(
                cal['year'], cal['mnt'], cal['day'], hour, minute
            ))

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    # {"day_name": "Thursday", "YEAR": {"current": 2024, "range": [2022, 2032]}, "day_in_mnt": 31, "mnt_num": 3, "mnt_name": "March", "day_num": 14}
    def get_calendar(self):
        try:
            now = datetime.now()
            cal = {
                "day_name": "Thursday",
                "YEAR": { "current": now.year, "range": [2022, 2032] },
                "day_in_mnt": calendar.monthrange(now.year, now.month)[1],
                "mnt_num": now.month,
                "mnt_name": calendar.month_name[now.month],
                "day_num": now.day
            }

            return cal

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())
            return {}

    # {"time_zone": ["UTC", "UTC"], "mnt": 14, "hour": 9}
    def set_time(self, time):
        try:
            now = datetime.now()
            year = now.year
            month = str(now.month).zfill(2)
            day = str(now.day).zfill(2)
            hour = str(time['hour']).zfill(2)
            minute = str(time['mnt']).zfill(2)

            subprocess.getoutput('date --set "{}-{}-{} {}:{}:00" && hwclock -w && timedatectl --adjust-system-clock'.format(year, month, day, hour, minute))

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_time(self):
        try:
            now = datetime.now()
            return { "time_zone": ["UTC", "UTC"], "mnt": now.minute, "hour": now.hour }

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())
            return {}

    def get_bat_level(self):
        try:
            # E6 05 5B 00 ChS
            self.send_device_cmd("e6055b0000")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_storage_percent(self):
        try:
            total, used, free = shutil.disk_usage('/media/')
            return (free / total) * 100

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())
            return 0

    def set_gps(self, gps):
        try:
            if gps:
                self.send_device_cmd("e60661010100")
                self.base_obj['GPS']['current'] = "ON"

            else:
                self.send_device_cmd("e60661010200")
                self.base_obj['GPS']['current'] = "OFF"

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_gps(self):
        try:
            # E6 06 61 01 00 ChS
            self.send_device_cmd("e60661010000")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_gps_position(self):
        try:
            gps_path = '/home/user/bidentifier/gpslocation'
            if os.path.exists(gps_path):
                with open(gps_path, 'r') as f:
                    coords = f.read().split(" ")
                    if len(coords) == 2: return float(coords[0]), float(coords[1])

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

        return 0.00, 0.00

    def set_dmc(self, dmc):
        try:
            if dmc:
                self.send_device_cmd("e60665040100")
                self.base_obj['DMC']['current'] = "ON"

            else:
                self.send_device_cmd("e60665040000")
                self.base_obj['DMC']['current'] = "OFF"

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_lrf_dmc_trg(self):
        try:
            # E6 05 59 00 ChS
            self.send_device_cmd("e605690000")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def set_target_lock(self, tgt_lock):
        try:
            if tgt_lock:
                self.send_device_cmd("e60668010100")
                self.base_obj['TARGET_LOCK']['current'] = "ON"
                self.base_obj['GPS']['current'] = "ON"
                self.base_obj['DMC']['current'] = "ON"

            else:
                self.send_device_cmd("e60668010000")
                self.base_obj['TARGET_LOCK']['current'] = "OFF"
                self.base_obj['GPS']['current'] = "OFF"
                self.base_obj['DMC']['current'] = "OFF"

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def target_lock_confirm(self):
        try:
            self.send_device_cmd("e605680300")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def target_lock_refuse(self):
        try:
            self.send_device_cmd("e605680200")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_target_lock_list(self):
        try:
            tgt_lock_path = '/bdr/tracking.log'
            if os.path.exists(tgt_lock_path):
                with open(tgt_lock_path, 'r', encoding='latin1') as f:
                    content = remove_non_ascii(f.read())
                    targets = content.split("\n")
                    return targets

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

        return []

    def lrf_measure(self):
        try:

            # Block status until LRF data is received
            self.block_status()

            self.send_device_cmd("e605670100")
            sleep(1)
            self.send_device_cmd("e605670200")

            # Get distance
            sleep(3.5)
            self.get_lrf_data()
            sleep(0.5)
            self.get_lrf_data()

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_lrf_data(self):
        try:
            # E6 05 5f 00 ChS
            self.send_device_cmd("e6055f0000")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    # Command  <hex> - Set LRF Unit
    # <E6>
    # <06>
    # <67>      // Command code LRF
    # <03>      // Set Unit
    # <value>   // Unit - 1/Meter , 2/Yard
    # <Chsum>
    # Response <hex> -<E6><05><01><00><chsum>
    def set_lrf_unit(self, unit):
        try:
            self.send_device_cmd("e6066703{}00".format(format(unit & 0xFF, '02x')))
            if unit == 1:
                self.base_obj['LRF']['UNT']['current'] = "METER"
            elif unit == 2:
                self.base_obj['LRF']['UNT']['current'] = "YARD"

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def set_auto_nuc_interval(self, interval):
        try:
            self.send_device_cmd("e60530{}00".format(format(interval & 0xFF, '02x')))

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_auto_nuc_interval(self):
        try:
            # E6 05 30 00 ChS
            self.send_device_cmd("e605300000")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_status(self):
        try:
            # E6 05 5C 00 ChS
            self.send_device_cmd("e6055c0000")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def set_display(self, display):
        try:

            # Display on
            if display:
                # e6 05 50 02 ChS
                # self.send_device_cmd("e605500200")
                subprocess.getoutput('/usr/bin/idroled enable 1')
            # Display off
            else:
                # e6 05 50 01 ChS
                # self.send_device_cmd("e605500100")
                subprocess.getoutput('/usr/bin/idroled enable 0')

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    ### Zoom ###

    def zoom_in(self):
        try:
            # e6 05 a1 01 43
            self.send_device_cmd("e605a10143")
            self.base_obj['ZOOM']['current'] = "CONT_ZOOM"

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def zoom_out(self):
        try:
            # e6 05 a1 00 42
            self.send_device_cmd("e605a10042")
            self.base_obj['ZOOM']['current'] = "CONT_ZOOM"

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def zoom_stop(self):
        try:
            # e6 05 a1 03 43
            self.send_device_cmd("e605a10343")
            self.base_obj['ZOOM']['current'] = "CONT_ZOOM"

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def digital_zoom(self, zoom):
        try:
            if zoom == "ZOOM_2":
                # e6 05 a1 02 40
                self.send_device_cmd("e605a10240")

            elif zoom == "ZOOM_4":
                # e6 05 a1 04 46
                self.send_device_cmd("e605a10446")

            self.base_obj['ZOOM']['current'] = zoom

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def get_zoom(self):
        try:
            # e6 05 a1 0a 48
            self.send_device_cmd("e605a10a48")

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    ### Authentication ###

    def auth_pin(self, pin):
        try:
            # Blcock status until response is received
            self.block_status()

            cmd = "e6" + format(6 + len(pin), '02x') + "a000" + format(len(pin), '02x')
            for char in pin: cmd += format(ord(char), '02x')
            cmd += "00"

            # e6 0a a0 00 04 30 30 30 30 ChS
            self.send_device_cmd(cmd)

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    def change_pin(self, old_pin, new_pin):
        try:
            # Blcock status until response is received
            self.block_status()

            # e6 11 a0 04 04 30 30 30 30 06 31 32 31 32 33 33 51
            cmd = "e6" + format(6 + len(old_pin) + 1 + len(new_pin), '02x') + "a004" + format(len(old_pin), '02x')
            for char in old_pin: cmd += format(ord(char), '02x')
            cmd += format(len(new_pin), '02x')
            for char in new_pin: cmd += format(ord(char), '02x')
            cmd += "00"

            self.send_device_cmd(cmd)

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

    # def reset_pin(self):
    #     try:
    #         # e6 10 a0 05 05 58 69 74 70 4f 04 30 30 30 30 28
    #         self.send_device_cmd("e610a00505586974704f043030303028")

    #     except Exception as e:
    #         printout(str(e))
    #         printout(traceback.format_exc())

    def get_base_object(self):
        try:
            return {
                "MODEL": "BDR 3.0 100",
                "BRT": {
                    "step": 5,
                    "range": [ -50, 50 ],
                    "current": 0
                },
                "CONT": {
                    "step": 5,
                    "range": [ 50, 150 ],
                    "current": 66
                },
                "WEATHER": {
                    "type_list": [ "hot", "normal", "wet" ],
                    "current": "hot"
                },
                "FILTER": {
                    "type_list": [ "on", "off" ],
                    "current": "off"
                },
                "PALETTE": {
                    "current": "WH",
                    "type_list": [ "BH", "WH", "RH", "Rainbow", "Ironbow", "Isotherm" ],
                    "isotherm": { "current": 0, "range": [ 1, 241 ] }
                },
                "ZOOM": {
                    "type_list": [ "ZOOM_1", "ZOOM_2", "ZOOM_4" ],
                    "current": "ZOOM_1"
                },
                "NUC_TYPE": {
                    "time": 2,
                    "type_list": [ "INT", "EXT" ],
                    "current": "EXT"
                },
                "LRF": {
                    "type_list": ["WAKE", "SLEEP"],
                    "current": "WAKE",
                    "distances": [],
                    "UNT": {
                        "type_list": [ "METER", "YARD" ],
                        "current": "METER"
                    }
                },
                "GPS": {
                  "type_list": ["ON", "OFF"],
                  "current": "OFF",
                  "position": []
                },
                "DMC": {
                  "type_list": ["ON", "OFF"],
                  "current": "OFF"
                },
                "TARGET_LOCK": {
                  "type_list": ["ON", "OFF"],
                  "current": "OFF"
                },
                "STANDBY": {
                    "type_list": [ "OFF", "ON" ],
                    "current": "OFF"
                },
                "BATT": { "current": 5, "range": [ 1, 7 ], }
             }

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())

            return {}
