import os, sys, tempfile, traceback, threading, shutil, asyncio
import signal as sig
from time import sleep

from app.utility import *
from app.core.bidentifier import *
from app.network.server.broadcast import *
from app.network.server.websocket.server import *
from app.network.server.websocket.handle import *


class Manager(object):

    def __init__(self):
        self.root_dir = os.path.abspath(os.path.dirname(__file__))
        self.version = open(os.path.join(self.root_dir, "version.txt"), 'r').read()
        
        # signal for the main loop to close application
        self.exit = False
        
        self.subtasks = []
        self.loop = None
        
        # Websocket
        self.ws_server = None
        self.ws_handle = None
        self.ws_clients = {}

        # Managers
        self.bidentifier = Bidentifier(self)
        
        # Initialize
        self.initialize()
                

    # Initialize
    def initialize(self):
        try:
            ### Websocket ###
            self.ws_handle = WsHandle(self)
            self.ws_server = WebSocketServer('', 8000, WBHandler, self, select_interval=0.1)
            threading.Thread(target=self.ws_server.serveforever, daemon=True).start()
            printout("Websocket: Server started...")
            
            # Network Init
            threading.Thread(target=self.network_init, daemon=True).start()
            
            # Run forever
            threading.Thread(target=self.run_until_closed, daemon=True).start()
            
        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())  
            
    # Network init
    def network_init(self):
        try:
            # new loop for the current thread
            if sys.version_info < (3, 10):
                asyncio.set_event_loop(asyncio.new_event_loop())
                self.loop = asyncio.get_event_loop()
                
            else:
                try:
                    self.loop = asyncio.get_running_loop()
                except RuntimeError:
                    self.loop = asyncio.new_event_loop()

                asyncio.set_event_loop(self.loop)

            # Broadcast (Discovery)
            self.subtasks.append(self.loop.create_task(self.loop.create_datagram_endpoint(lambda: Broadcast(self), local_addr=('0.0.0.0', 7999))))

            self.loop.run_until_complete(asyncio.wait(self.subtasks))
            self.loop.run_forever()
            self.loop.close()

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())
     
    # Loop the manager
    def run_until_closed(self):
        try:
            while not self.exit: sleep(0.1)

            # Close websocket server
            self.ws_server.close()

        except Exception as e:
            printout(str(e))
            printout(traceback.format_exc())  


class Application(object):
    manager = None
    
    def _exit(self):
        os.remove(self._pid_file)
        self.manager.exit = True

    def _signal_handler(self, signal, frame):
        self._exit()
        sys.exit(0)

    def start(self):
        self._pid_file = os.path.join(tempfile.gettempdir(), "axion-translator.pid")

        sig.signal(sig.SIGINT, self._signal_handler)
        sig.signal(sig.SIGTERM, self._signal_handler)
        sig.signal(sig.SIGILL, self._signal_handler)
        sig.signal(sig.SIGSEGV, self._signal_handler)
        sig.signal(sig.SIGABRT, self._signal_handler)
        sig.signal(sig.SIGHUP, self._signal_handler)

        # Common Object
        self.manager = Manager()
        
        # Title
        title = "Axion Translator - {}".format(self.manager.version)
        (width, height) = shutil.get_terminal_size()
        printout("\n" + ''.center(width, '*') + "\n" + title + "\n" + ''.center(width, '*') + "\r\n")

        try:
            if os.path.exists(self._pid_file): os.remove(self._pid_file)
            with open(self._pid_file, mode="w") as file: file.write(str(os.getpid()))

            printout("Ctrl-C to exit...")
            while True: sleep(1.0)

        except KeyboardInterrupt:
            self._exit

        printout(''.center(width, '*'))
        printout(title + " stopped!")


if __name__ == "__main__":
    app = Application()
    app.start()
